package br.com.mzsw;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Biblioteca para conexo com balanas.
 * obtm o peso de balanas, realiza conexo automtica
 * 
 * @author Mazin
 *
 */
public class PesoLib implements Runnable {
	private List<BalancaListener> listeners;
	private PesoLibWrapper driver;
	private long instance;
	private boolean canceled;
	private int ultimoPeso;
	
	/**
	 * Cria uma conexo com uma balana
	 * 
	 * @param configuracao
	 */
	public PesoLib() {
		listeners = new ArrayList<>();
		driver = new PesoLibWrapper();
		instance = driver.criar("");
		if(instance == 0)
			throw new RuntimeException("No foi possvel criar uma instncia da biblioteca");
		Thread thread = new Thread(this);
		thread.start();
	}
	
	/**
	 * Cria uma conexo com uma balana especificando configurao de porta e tempo
	 * 
	 * @param configuracao
	 */
	public PesoLib(String configuracao) {
		listeners = new ArrayList<>();
		driver = new PesoLibWrapper();
		instance = driver.criar(configuracao);
		if(instance == 0)
			throw new RuntimeException("No foi possvel criar uma instncia da biblioteca");
		Thread thread = new Thread(this);
		thread.start();
	}
	
	/**
	 * Adiciona uma interface que receber evento de conexo e recebimento de peso
	 * 
	 * @param l interface que receber os eventos
	 */
	public void addEventListener(BalancaListener l) {
		listeners.add(l);
	}
	
	/**
	 * Fecha a conexo com a balana, aps a chamada desse mtodo, a instancia da classe no poder mais ser usada
	 * 
	 */
	public void fecha() {
		if(canceled)
			return;
		canceled = true;
		driver.cancela(instance);
		driver.libera(instance);
		instance = 0;
	}
	
	private void needActive() {
		if(canceled)
			throw new RuntimeException("A instancia da biblioteca j foi liberada");	
	}
	
	/**
	 * Informa para a balana o preo do item que est sendo pesado
	 * 
	 * @param preco preo do item
	 */
	public void setPreco(float preco) {
		needActive();
		if(!driver.solicitaPeso(instance, preco))
			throw new RuntimeException("No foi possvel ajustar o preo do item da balaa");
	}
	
	/**
	 * Solicita o peso do item sobre a balana
	 * 
	 */
	public void askPeso() {
		setPreco(0.0f);
	}

	/**
	 * Informa se est conectado  uma balana
	 * 
	 * @return true se est conectado, falso caso contrrio
	 */
	public boolean isConectado() {
		if(canceled)
			return true;
		return driver.isConectado(instance);
	}
	
	/**
	 * Ajusta configuraes de conexo com a balana e tempo de espera
	 * 
	 * @param configuracao configuraes com instrues separadas por ;
	 * exemplo: port:COM3;baund:9600, ajusta a porta e a velocidade de conexo
	 */
	public void setConfiguracao(String configuracao) {
		needActive();
		driver.setConfiguracao(instance, configuracao);
	}
	
	/**
	 * Obtm a configurao da conexo atual
	 * 
	 * @return configuraes com instrues separadas por ;
	 * exemplo: port:COM3;baund:9600, o primero  a porta e segundo  a velocidade de conexo 
	 */
	public String getConfiguracao() {
		needActive();
		return driver.getConfiguracao(instance);
	}
	

	/**
	 * Obtm o ltimo peso enviado pela balana
	 * 
	 * @return peso em gramas
	 */
	public int getUltimoPeso() {
		return ultimoPeso;
	}

	/**
	 * Obtm todas as marcas de balanas suportadas pela biblioteca
	 * 
	 * @return lista com as marcas suportadas
	 */
	public List<String> getMarcas() {
		needActive();
		String marcas = driver.getMarcas(instance);
		return Arrays.asList(marcas.split("\r\n"));
	}
	
	/**
	 * Obtm todas os modelos suportados pela balana da marca informada
	 * 
	 * @param marca marca da balana
	 * @return lista com todos os modelos suportados
	 */
	public List<String> getModelos(String marca) {
		needActive();
		String modelos = driver.getModelos(instance, marca);
		return Arrays.asList(modelos.split("\r\n"));
	}
	
	/**
	 * Obtm a verso da biblioteca
	 * 
	 * @return verso no formato 0.0.0.0
	 */
	public String getVersao() {
		return driver.getVersao(instance);
	}
	
	@Override
	public void run() {
		int event;
		
		do {
			event = driver.aguardaEvento(instance);
			switch (event) {
			case PesoLibWrapper.EVENTO_CANCELADO:
				break;
			case PesoLibWrapper.EVENTO_CONECTADO:
				postEventConnect();
				break;
			case PesoLibWrapper.EVENTO_DESCONECTADO:
				postEventDisconnect();
				break;
			default:
				postEventWeightReceived(driver.getUltimoPeso(instance));
			}
		} while(!canceled && event != PesoLibWrapper.EVENTO_CANCELADO);
	}

	private void postEventConnect() {
		for (BalancaListener elem : listeners) {
			elem.onConectado(this);
		}
	}

	private void postEventDisconnect() {
		for (BalancaListener elem : listeners) {
			elem.onDesconectado(this);
		}
	}

	private void postEventWeightReceived(int ultimoPeso) {
		this.ultimoPeso = ultimoPeso;
		for (BalancaListener elem : listeners) {
			elem.onPesoRecebido(this, ultimoPeso);
		}
	}

}
